<?php
require_once 'admin_config.php';
requireAdminAuth(); // Ensure only admins can access this page

$user = null;
$success = false;
$error = '';

// Get user ID from URL
$user_id = isset($_GET['id']) ? intval($_GET['id']) : 0;

// Fetch existing user data
if ($user_id > 0) {
    try {
        $conn = getAdminDBConnection();
        $stmt = $conn->prepare("SELECT id, username, full_name, email, balance, role FROM users WHERE id = ?");
        if ($stmt === false) {
            throw new Exception('Prepare fetch statement failed: ' . $conn->error);
        }
        $stmt->bind_param('i', $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result->num_rows > 0) {
            $user = $result->fetch_assoc();
        } else {
            $error = 'المستخدم المطلوب غير موجود.';
        }
        $stmt->close();
    } catch (Exception $e) {
        $error = 'خطأ أثناء تحميل بيانات المستخدم: ' . $e->getMessage();
    }
} else {
    $error = 'معرف المستخدم غير صالح.';
}

// Handle form submission for updating a user
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $user_id > 0 && $user) {
    $username = trim($_POST['username'] ?? '');
    $full_name = trim($_POST['full_name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $balance = floatval($_POST['balance'] ?? 0);
    $role = $user['role']; // Keep existing role, don't allow changes
    $password = $_POST['password'] ?? ''; // Optional: only update if provided

    if (empty($username) || empty($email)) {
        $error = 'يرجى ملء الحقول المطلوبة (اسم المستخدم، البريد الإلكتروني).';
    } else if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = 'صيغة البريد الإلكتروني غير صحيحة.';
    } else {
        try {
            $conn = getAdminDBConnection();

            // Check for duplicate username/email, excluding current user
            $stmt_check = $conn->prepare("SELECT id FROM users WHERE (username = ? OR email = ?) AND id != ?");
            if ($stmt_check === false) {
                throw new Exception('Prepare check statement failed: ' . $conn->error);
            }
            $stmt_check->bind_param('ssi', $username, $email, $user_id);
            $stmt_check->execute();
            $stmt_check->store_result();
            if ($stmt_check->num_rows > 0) {
                $error = 'اسم المستخدم أو البريد الإلكتروني موجود بالفعل لمستخدم آخر.';
            } else {
                $sql = "UPDATE users SET username = ?, full_name = ?, email = ?, balance = ?, role = ?, updated_at = NOW()";
                $params = [$username, $full_name, $email, $balance, $role];
                $types = 'sssds';

                if (!empty($password)) {
                    $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                    $sql .= ", password = ?";
                    $params[] = $hashed_password;
                    $types .= 's';
                }

                $sql .= " WHERE id = ?";
                $params[] = $user_id;
                $types .= 'i';

                $stmt = $conn->prepare($sql);
                if ($stmt === false) {
                    throw new Exception('Prepare update statement failed: ' . $conn->error);
                }

                $stmt->bind_param($types, ...$params);

                if ($stmt->execute()) {
                    $message = 'تم تحديث المستخدم بنجاح!';
                    $message_type = 'success';
                    header('Location: admin_users.php?message=' . urlencode($message) . '&type=' . $message_type);
                    exit();
                } else {
                    $error = 'حدث خطأ أثناء تحديث المستخدم: ' . $conn->error;
                }
                $stmt->close();
            }
            $stmt_check->close();
        } catch (Exception $e) {
            $error = 'خطأ: ' . $e->getMessage();
        }
    }
}

require_once '../header.php'; // Include header for consistent styling
?>

<div class="container bg-dark text-white mt-5">
    <div class="card">
        <div class="card-header">
            <h3 class="mb-0">تعديل المستخدم: <?= htmlspecialchars($user['username'] ?? '') ?></h3>
        </div>
        <div class="card-body">
            <?php if (!empty($message)): ?>
                <div class="alert alert-<?= htmlspecialchars($message_type) ?>" role="alert">
                    <?= htmlspecialchars($message) ?>
                </div>
            <?php elseif ($error): ?>
                <div class="alert alert-danger" role="alert">
                    <?= htmlspecialchars($error) ?>
                </div>
            <?php endif; ?>

            <?php if ($user): ?>
            <form method="POST">
                <div class="mb-3">
                    <label for="username" class="form-label">اسم المستخدم <span class="text-danger">*</span></label>
                    <input type="text" class="form-control" id="username" name="username" value="<?= htmlspecialchars($user['username'] ?? '') ?>" required>
                </div>
                <div class="mb-3">
                    <label for="full_name" class="form-label">الاسم الكامل</label>
                    <input type="text" class="form-control" id="full_name" name="full_name" value="<?= htmlspecialchars($user['full_name'] ?? '') ?>">
                </div>
                <div class="mb-3">
                    <label for="email" class="form-label">البريد الإلكتروني <span class="text-danger">*</span></label>
                    <input type="email" class="form-control" id="email" name="email" value="<?= htmlspecialchars($user['email'] ?? '') ?>" required>
                </div>
                <div class="mb-3">
                    <label for="password" class="form-label">كلمة المرور (اترك فارغاً لعدم التغيير)</label>
                    <input type="password" class="form-control" id="password" name="password">
                </div>
                <div class="mb-3">
                    <label for="balance" class="form-label">الرصيد</label>
                    <input type="number" step="0.01" class="form-control" id="balance" name="balance" value="<?= htmlspecialchars($user['balance'] ?? '0') ?>">
                </div>
                <div class="mb-3">
                    <label for="role" class="form-label">الدور</label>
                    <select class="form-select" id="role" name="role" disabled>
                        <option value="user" <?= (($user['role'] ?? '') === 'user') ? 'selected' : '' ?>>مستخدم</option>
                        <option value="admin" <?= (($user['role'] ?? '') === 'admin') ? 'selected' : '' ?>>أدمن</option>
                    </select>
                    <small class="text-muted">لا يمكن تغيير دور المستخدم</small>
                </div>
                <button type="submit" class="btn btn-primary">تحديث المستخدم</button>
                <a href="admin_users.php" class="btn btn-secondary">إلغاء</a>
            </form>
            <?php endif; ?>
        </div>
    </div>
</div>

<?php require_once '../footer.php'; ?> 